/**
 * Copyright IBM Corp. 2018, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

'use strict';

const { builders } = require('@carbon/icon-build-helpers');
const metadata = require('@carbon/pictograms/metadata.json');
const fs = require('fs-extra');
const path = require('path');
const ts = require('typescript');

const BANNER = `/**
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * Code generated by @carbon/pictograms-react. DO NOT EDIT.
 */`;

async function build() {
  await builders.react.run(metadata, {
    output: process.cwd(),
  });

  await generateTypeScriptDefinitions();
}

async function generateTypeScriptDefinitions() {
  const targets = [
    {
      directory: path.join(process.cwd(), 'es'),
      moduleKind: ts.ModuleKind.ESNext,
    },
    {
      directory: path.join(process.cwd(), 'lib'),
      moduleKind: ts.ModuleKind.CommonJS,
    },
  ];

  for (const target of targets) {
    await compileIconComponent(target.directory, target.moduleKind);
    await copyCarbonPictogramType(target.directory);
    await generateModuleTypes(metadata, target.directory);
    await generateIndexTypes(target.directory);
  }
}

async function compileIconComponent(outDir, moduleKind) {
  const iconComponentPath = path.resolve(
    __dirname,
    '../../icon-build-helpers/src/builders/react/components/Icon.tsx'
  );
  const options = {
    module: moduleKind,
    outDir,
    declaration: true,
    emitDeclarationOnly: true,
    skipLibCheck: true,
    jsx: ts.JsxEmit.React,
    esModuleInterop: true,
    target: ts.ScriptTarget.ES2015,
    moduleResolution: ts.ModuleResolutionKind.NodeJs,
    lib: ['lib.es2015.d.ts', 'lib.dom.d.ts'],
    types: ['react'],
  };

  const host = ts.createCompilerHost(options);
  const program = ts.createProgram([iconComponentPath], options, host);
  const emitResult = program.emit();

  const diagnostics = ts
    .getPreEmitDiagnostics(program)
    .concat(emitResult.diagnostics);
  if (diagnostics.length > 0) {
    diagnostics.forEach((diagnostic) => {
      if (diagnostic.file) {
        const { line, character } =
          diagnostic.file.getLineAndCharacterOfPosition(diagnostic.start);
        const message = ts.flattenDiagnosticMessageText(
          diagnostic.messageText,
          '\n'
        );
        console.log(
          `${diagnostic.file.fileName} (${line + 1},${character + 1}): ${message}`
        );
      } else {
        console.log(
          ts.flattenDiagnosticMessageText(diagnostic.messageText, '\n')
        );
      }
    });
  }
}

async function copyCarbonPictogramType(outDir) {
  const srcPath = path.resolve(
    __dirname,
    '../../icon-build-helpers/src/builders/react/components/CarbonPictogram.d.ts'
  );
  const destPath = path.resolve(outDir, 'CarbonPictogram.d.ts');
  await fs.copyFile(srcPath, destPath);
}

async function generateModuleTypes(metadata, outDir) {
  for (const icon of metadata.icons) {
    for (const output of icon.output) {
      const filepath = output.filepath.replace(/\.js$/, '.d.ts');
      const fileDir = path.dirname(filepath);
      const depth = fileDir.split(path.sep).length;
      const relativePath =
        depth > 0
          ? '../'.repeat(depth) + 'CarbonPictogram'
          : './CarbonPictogram';

      const content =
        BANNER +
        '\n' +
        `import type { CarbonPictogramType } from '${relativePath}';\n` +
        `declare const ${output.moduleName}: CarbonPictogramType;\n` +
        `export default ${output.moduleName}\n`;

      const filename = path.resolve(outDir, filepath);
      await fs.ensureDir(path.dirname(filename));
      await fs.writeFile(filename, content);
    }
  }
}

async function generateIndexTypes(outDir) {
  const indexJsPath = path.resolve(outDir, 'index.js');
  const indexContent = await fs.readFile(indexJsPath, 'utf8');

  const bucketExports = [];
  const esmRegex = /from '\.\/(__generated__\/bucket-\d+)\.js'/g;
  const cjsRegex = /require\('\.\/(__generated__\/bucket-\d+)\.js'\)/g;

  let match;
  while ((match = esmRegex.exec(indexContent)) !== null) {
    if (!bucketExports.includes(match[1])) {
      bucketExports.push(match[1]);
    }
  }
  while ((match = cjsRegex.exec(indexContent)) !== null) {
    if (!bucketExports.includes(match[1])) {
      bucketExports.push(match[1]);
    }
  }

  for (const bucketPath of bucketExports) {
    await generateBucketTypes(metadata, outDir, bucketPath);
  }

  const indexContent_dts =
    BANNER +
    '\n' +
    "export { default as Icon } from './Icon';\n" +
    "export type { CarbonPictogramProps, CarbonPictogramType } from './CarbonPictogram';\n" +
    bucketExports.map((path) => `export * from './${path}';`).join('\n') +
    '\n';

  await fs.writeFile(path.resolve(outDir, 'index.d.ts'), indexContent_dts);
}

async function generateBucketTypes(metadata, outDir, bucketPath) {
  const bucketJsPath = path.resolve(outDir, bucketPath + '.js');
  const bucketContent = await fs.readFile(bucketJsPath, 'utf8');

  const exports = [];
  const exportLines = bucketContent.match(/^export \{[^}]+\}/gm);

  if (exportLines) {
    for (const line of exportLines) {
      const namesMatch = line.match(/\{([^}]+)\}/);
      if (namesMatch) {
        const names = namesMatch[1]
          .split(',')
          .map((n) => n.trim())
          .filter((n) => n);
        exports.push(...names);
      }
    }
  }

  const iconLines = exports.map(
    (name) => `declare const _${name}: CarbonPictogramType;`
  );
  const exportList = exports.map((name) => `_${name} as ${name}`);

  const content =
    BANNER +
    '\n' +
    "import type { CarbonPictogramType } from '../CarbonPictogram';\n" +
    iconLines.join('\n') +
    `\nexport { ${exportList.join(', ')} };\n`;

  const filename = path.resolve(outDir, bucketPath + '.d.ts');
  await fs.ensureDir(path.dirname(filename));
  await fs.writeFile(filename, content);
}

build().catch((error) => {
  console.error(error);
  process.exit(1);
});
