/**
 * Copyright IBM Corp. 2016, 2025
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import PropTypes from 'prop-types';
import React, { type HTMLAttributes } from 'react';
import cx from 'classnames';
import { usePrefix } from '../../internal/usePrefix';

export interface FormGroupProps extends HTMLAttributes<HTMLFieldSetElement> {
  /**
   * Provide the children form elements to be rendered inside of the <fieldset>
   */
  children: React.ReactNode;
  /**
   * Provide a custom className to be applied to the containing <fieldset> node
   */
  className?: string;
  /**
   * Specify whether the FormGroup should be disabled
   */
  disabled?: boolean;
  /**
   * Specify whether the <FormGroup> is invalid
   */
  invalid?: boolean;
  /**
   * Provide id for the fieldset <legend> which corresponds to the fieldset
   * `aria-labelledby`
   */
  legendId?: string;
  /**
   * Provide the text to be rendered inside of the fieldset <legend>
   */
  legendText: React.ReactNode;

  /**
   * Specify whether the message should be displayed in the <FormGroup>
   */
  message?: boolean;
  /**
   * Provide the text for the message in the <FormGroup>
   */
  messageText?: string;
}

const FormGroup = ({
  disabled = false,
  legendId,
  legendText,
  invalid = false,
  children,
  className,
  message = false,
  messageText = '',
  ...rest
}: FormGroupProps) => {
  const prefix = usePrefix();

  const classNamesFieldset = cx(`${prefix}--fieldset`, className);

  return (
    <fieldset
      disabled={disabled}
      {...(invalid && { 'data-invalid': '' })}
      className={classNamesFieldset}
      {...rest}
      aria-labelledby={rest['aria-labelledby'] || legendId}>
      <legend
        className={`${prefix}--label`}
        id={legendId || rest['aria-labelledby']}>
        {legendText}
      </legend>
      {children}
      {message ? (
        <div className={`${prefix}--form__requirements`}>{messageText}</div>
      ) : null}
    </fieldset>
  );
};

FormGroup.propTypes = {
  /**
   * Provide the children form elements to be rendered inside of the <fieldset>
   */
  children: PropTypes.node,

  /**
   * Provide a custom className to be applied to the containing <fieldset> node
   */
  className: PropTypes.string,

  /**
   * Specify whether the FormGroup should be disabled
   */
  disabled: PropTypes.bool,

  /**
   * Specify whether the <FormGroup> is invalid
   */
  invalid: PropTypes.bool,

  /**
   * Provide id for the fieldset <legend> which corresponds to the fieldset
   * `aria-labelledby`
   */
  legendId: PropTypes.node,

  /**
   * Provide the text to be rendered inside of the fieldset <legend>
   */
  legendText: PropTypes.node.isRequired,

  /**
   * Specify whether the message should be displayed in the <FormGroup>
   */
  message: PropTypes.bool,

  /**
   * Provide the text for the message in the <FormGroup>
   */
  messageText: PropTypes.string,
};

export default FormGroup;
