/**
 * Copyright IBM Corp. 2019, 2025
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings';
import { forEach, indexOf } from '../../globals/internal/collection-helpers';
import { NAVIGATION_DIRECTION, CONTENT_SWITCHER_SIZE } from './defs';
import CDSContentSwitcherItem from './content-switcher-item';
import styles from './content-switcher.scss?lit';
import { carbonElement as customElement } from '../../globals/decorators/carbon-element';

export { NAVIGATION_DIRECTION, CONTENT_SWITCHER_SIZE };

/**
 * @param index The index
 * @param length The length of the array.
 * @returns The new index, adjusting overflow/underflow.
 */
const capIndex = (index: number, length: number) => {
  if (index < 0) {
    return length - 1;
  }
  if (index >= length) {
    return 0;
  }
  return index;
};

/**
 * Content switcher.
 *
 * @element cds-content-switcher
 * @fires cds-content-switcher-beingselected
 *   The custom event fired before a content switcher item is selected upon a user gesture.
 *   Cancellation of this event stops changing the user-initiated selection.
 * @fires cds-content-switcher-selected - The custom event fired after a a content switcher item is selected upon a user gesture.
 */
@customElement(`${prefix}-content-switcher`)
class CDSContentSwitcher extends LitElement {
  /**
   * Handles `mouseover`/`mouseout` events on `<slot>`.
   *
   * @param event The event.
   * @param event.target The event target.
   * @param event.type The event type.
   */
  private _handleHover({ target, type }: MouseEvent) {
    const { selectorItem } = this.constructor as typeof CDSContentSwitcher;
    const items = this.querySelectorAll(selectorItem);
    const index =
      type !== 'mouseover'
        ? -1
        : // eslint-disable-next-line @typescript-eslint/no-non-null-assertion -- https://github.com/carbon-design-system/carbon/issues/20452
          indexOf(items, (target as Element).closest(selectorItem)!);

    if ((target as Element).closest(selectorItem)?.hasAttribute('disabled')) {
      return;
    }

    const nextIndex = index < 0 ? index : index + 1;
    forEach(this.querySelectorAll(selectorItem), (elem, i) => {
      // Specifies child `<cds-content-switcher-item>` to hide its divider instead of using CSS,
      // until `:host-context()` gets supported in all major browsers
      (elem as CDSContentSwitcherItem).hideDivider = i === nextIndex;
    });

    const { selectorItemSelected } = this
      .constructor as typeof CDSContentSwitcher;
    const selectedItem = this.querySelector(selectorItemSelected);
    const nextItem = this._getNextItem(
      selectedItem as CDSContentSwitcherItem,
      1
    );
    (nextItem as CDSContentSwitcherItem).hideDivider = true;
  }

  /**
   * @param target The current event target.
   * @returns The item to be selected.
   */
  protected _getCurrentItem(target: HTMLElement) {
    const items = this.querySelectorAll(
      (this.constructor as typeof CDSContentSwitcher).selectorItemEnabled
    );
    const { selectorItem } = this.constructor as typeof CDSContentSwitcher;
    const containerItem = target.closest(
      selectorItem
    ) as CDSContentSwitcherItem;
    const index = indexOf(items, containerItem);
    return items[index] ?? null;
  }

  /**
   * @param currentItem The currently selected item.
   * @param direction The navigation direction.
   * @returns The item to be selected.
   */
  protected _getNextItem(
    currentItem: CDSContentSwitcherItem,
    direction: number
  ) {
    const items = this.querySelectorAll(
      (this.constructor as typeof CDSContentSwitcher).selectorItemEnabled
    );
    const currentIndex = indexOf(items, currentItem);
    const nextIndex = capIndex(currentIndex + direction, items.length);
    return nextIndex === currentIndex ? null : items[nextIndex];
  }

  /**
   * Handles `click` event on content switcher item.
   *
   * @param event The event.
   * @param event.target The event target.
   */
  protected _handleClick({ target }: MouseEvent) {
    const currentItem = this._getCurrentItem(target as HTMLElement);
    this._handleUserInitiatedSelectItem(
      currentItem as CDSContentSwitcherItem,
      'mouse'
    );
  }

  /**
   * Handles `keydown` event on the top-level element in the shadow DOM.
   *
   * @param event The event.
   * @param event.key The event key.
   */
  protected _handleKeydown({ key }: KeyboardEvent) {
    if (key in NAVIGATION_DIRECTION) {
      this._navigate(NAVIGATION_DIRECTION[key]);
    }
  }

  /**
   * Handles user-initiated selection of a content switcher item.
   *
   * @param [item] The content switcher item user wants to select.
   */
  protected _handleUserInitiatedSelectItem(
    item: CDSContentSwitcherItem,
    interactionType?: 'mouse' | 'keyboard' | undefined
  ) {
    if (
      (item && !item.disabled && item.value !== this.value) ||
      (this.selectionMode === 'manual' &&
        interactionType === 'keyboard' &&
        !item.disabled)
    ) {
      const init = {
        bubbles: true,
        composed: true,
        detail: {
          item,
        },
      };
      const constructor = this.constructor as typeof CDSContentSwitcher;
      const beforeSelectEvent = new CustomEvent(constructor.eventBeforeSelect, {
        ...init,
        cancelable: true,
      });
      if (this.dispatchEvent(beforeSelectEvent)) {
        this._selectionDidChange(item, interactionType);

        // Add extra event details (index, name, text) to match the React `onChange`
        const items = this.querySelectorAll(constructor.selectorItem);
        const index = Array.from(items).indexOf(item);
        const name = item.getAttribute('name') ?? undefined;
        const text = item.textContent?.trim() ?? undefined;

        const afterSelectEvent = new CustomEvent(constructor.eventSelect, {
          bubbles: true,
          composed: true,
          detail: {
            item,
            index,
            name,
            text,
          },
        });
        this.dispatchEvent(afterSelectEvent);
      }
    }
  }

  /**
   * Navigates through content switcher items.
   *
   * @param direction `-1` to navigate backward, `1` to navigate forward.
   */
  protected _navigate(direction: number) {
    const { selectorItemFocused } = this
      .constructor as typeof CDSContentSwitcher;
    const nextItem = this._getNextItem(
      this.querySelector(selectorItemFocused) as CDSContentSwitcherItem,
      direction
    );
    if (nextItem) {
      this._handleUserInitiatedSelectItem(
        nextItem as CDSContentSwitcherItem,
        'keyboard'
      );
      this.requestUpdate();
    }
  }

  /**
   * A callback that runs after change in content switcher selection upon user interaction is confirmed.
   *
   * @param itemToSelect A content switcher item.
   */

  protected _selectionDidChange(
    itemToSelect: CDSContentSwitcherItem,
    interactionType?: 'mouse' | 'keyboard' | undefined
  ) {
    if (this.selectionMode === 'manual' && interactionType === 'keyboard') {
      // In manual mode, only focus the item without changing the selection
      Promise.resolve().then(() => {
        itemToSelect.focus();
      });

      return;
    }

    this.value = itemToSelect.value;
    forEach(
      this.querySelectorAll(
        (this.constructor as typeof CDSContentSwitcher).selectorItemSelected
      ),
      (item) => {
        (item as CDSContentSwitcherItem).selected = false;
      }
    );
    itemToSelect.selected = true;
    // Waits for rendering with the new state that updates `tabindex`
    Promise.resolve().then(() => {
      itemToSelect.focus();

      const { selectorItem } = this.constructor as typeof CDSContentSwitcher;
      const items = this.querySelectorAll(selectorItem);
      const index = indexOf(
        items,
        (itemToSelect as Element).closest(selectorItem)! // eslint-disable-line @typescript-eslint/no-non-null-assertion -- https://github.com/carbon-design-system/carbon/issues/20452
      );
      const nextIndex = index < 0 ? index : index + 1;
      forEach(this.querySelectorAll(selectorItem), (elem, i) => {
        // Specifies child `<cds-content-switcher-item>` to hide its divider instead of using CSS,
        // until `:host-context()` gets supported in all major browsers
        (elem as CDSContentSwitcherItem).hideDivider = i === nextIndex;
      });
    });
  }

  /**
   * The value of the selected item.
   */
  @property({ reflect: true })
  value = '';

  /**
   * Specify a selected index for the initially selected content
   */
  @property({ type: Number, attribute: 'selected-index' })
  selectedIndex = 0;

  /**
   * Choose whether or not to automatically change selection on focus when left/right arrow pressed. Defaults to 'automatic'
   */
  @property({ attribute: 'selection-mode' })
  selectionMode = 'automatic';

  /**
   * Content switcher size.
   */
  @property({ reflect: true })
  size = CONTENT_SWITCHER_SIZE.REGULAR;

  /**
   * Icon only.
   */
  @property({ type: Boolean, reflect: true, attribute: 'icon' })
  iconOnly = false;

  /**
   * `true` to use the low contrast version.
   */
  @property({ type: Boolean, reflect: true, attribute: 'low-contrast' })
  lowContrast = false;

  firstUpdated() {
    this._updateSelectedItemFromIndex();
  }

  // Validate a selected index for the initially selected content
  _updateSelectedItemFromIndex() {
    const { selectorItemEnabled } = this
      .constructor as typeof CDSContentSwitcher;
    const items = this.querySelectorAll(selectorItemEnabled);
    if (
      items.length > 0 &&
      this.selectedIndex >= 0 &&
      this.selectedIndex < items.length
    ) {
      const itemToSelect = items[this.selectedIndex] as CDSContentSwitcherItem;
      this._selectionDidChange(
        itemToSelect,
        this.selectionMode === 'manual' ? 'keyboard' : 'mouse'
      );
    }
  }

  _updateSelectedItemFromValue(changedProps) {
    if (changedProps.has('value')) {
      const { selectorItem } = this.constructor as typeof CDSContentSwitcher;
      forEach(this.querySelectorAll(selectorItem), (elem) => {
        (elem as CDSContentSwitcherItem).selected =
          (elem as CDSContentSwitcherItem).value === this.value;
      });
    }
  }

  shouldUpdate(changedProps) {
    if (changedProps.has('iconOnly') || changedProps.has('selectedIndex')) {
      const items = this.querySelectorAll(`${prefix}-content-switcher-item`);
      const allIcon = Array.from(items).every((item) =>
        item.hasAttribute('icon')
      );
      this.iconOnly = allIcon;
    }
    return true;
  }

  updated(changedProperties) {
    if (changedProperties.has('selectedIndex')) {
      this._updateSelectedItemFromIndex();
    }

    this._updateSelectedItemFromValue(changedProperties);

    if (changedProperties.has('size')) {
      const items = this.querySelectorAll(`${prefix}-content-switcher-item`);
      items.forEach((item) => {
        const size = this.size || 'md';
        item.setAttribute('size', size);
      });
    }
  }

  _handleSlotchange() {
    const { selectorItemSelected } = this
      .constructor as typeof CDSContentSwitcher;
    const selectedItem = this.querySelector(selectorItemSelected);
    const nextItem = this._getNextItem(
      selectedItem as CDSContentSwitcherItem,
      1
    );

    // Specifies child `<cds-content-switcher-item>` to hide its divider instead of using CSS,
    // until `:host-context()` gets supported in all major browsers
    if (nextItem) {
      (nextItem as CDSContentSwitcherItem).hideDivider = true;
    }
  }

  /**
   * A selector that will return content switcher items.
   */
  static get selectorItem() {
    return `${prefix}-content-switcher-item`;
  }

  /**
   * A selector that will return content switcher icon items.
   */
  static get selectorIconItem() {
    return `${prefix}-content-switcher-item[icon]`;
  }

  /**
   * A selector that will return enabled content switcher items.
   */
  static get selectorItemEnabled() {
    return `${prefix}-content-switcher-item:not([disabled])`;
  }

  /**
   * A selector that will return selected items.
   */
  static get selectorItemSelected() {
    return `${prefix}-content-switcher-item[selected]`;
  }

  /**
   * A selector that will return focused items.
   */
  static get selectorItemFocused() {
    return `${prefix}-content-switcher-item:focus`;
  }

  /**
   * The name of the custom event fired before a content switcher item is selected upon a user gesture.
   * Cancellation of this event stops changing the user-initiated selection.
   */
  static get eventBeforeSelect() {
    return `${prefix}-content-switcher-beingselected`;
  }

  /**
   * The name of the custom event fired after a a content switcher item is selected upon a user gesture.
   */
  static get eventSelect() {
    return `${prefix}-content-switcher-selected`;
  }

  render() {
    const {
      _handleHover: handleHover,
      _handleKeydown: handleKeydown,
      _handleSlotchange: handleSlotchange,
    } = this;
    return html`
      <slot
        @click="${this._handleClick}"
        @keydown="${handleKeydown}"
        @mouseover="${handleHover}"
        @mouseout="${handleHover}"
        @slotchange=${handleSlotchange}></slot>
    `;
  }

  static styles = styles;
}

export default CDSContentSwitcher;
