/**
 * Copyright IBM Corp. 2019, 2026
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { html, render } from 'lit';
import CDSRadioButtonGroup, {
  RADIO_BUTTON_ORIENTATION,
} from '../../src/components/radio-button/radio-button-group';
import { RADIO_BUTTON_LABEL_POSITION } from '../../src/components/radio-button/radio-button';
import { Playground } from '../../src/components/radio-button/radio-button-story';

// JSDOM's DOM implementation does not provide `FormDataEvent`.
type FormDataEventLike = Event & { formData: FormData };

/**
 * @param formData A `FormData` instance.
 * @returns The given `formData` converted to a classic key-value pair.
 */
const getValues = (formData: FormData) => {
  const values = {};

  for (const [key, value] of formData.entries()) {
    values[key] = value;
  }
  return values;
};

const template = (props?) =>
  Playground({
    'cds-radio-button-group': props,
  });

xdescribe('cds-radio-button', () => {
  describe('Rendering', () => {
    it('Should render with minimum attributes', async () => {
      render(template(), document.body);
      await Promise.resolve();
      expect(
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- https://github.com/carbon-design-system/carbon/issues/20452
        document.body.querySelector('cds-radio-button[value="staging"]' as any)
      ).toMatchSnapshot({ mode: 'shadow' });
    });

    it('Should render with various attributes', async () => {
      render(
        template({
          'cds-radio-button-group': {
            disabled: true,
            labelPosition: RADIO_BUTTON_LABEL_POSITION.LEFT,
            name: 'name-foo',
            orientation: RADIO_BUTTON_ORIENTATION.VERTICAL,
            value: 'staging',
          },
          'cds-radio-button': {
            hideLabel: true,
            labelText: 'label-text-foo',
          },
        }),
        document.body
      );
      await Promise.resolve();
      expect(
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- https://github.com/carbon-design-system/carbon/issues/20452
        document.body.querySelector('cds-radio-button[value="staging"]' as any)
      ).toMatchSnapshot({ mode: 'shadow' });
    });
  });

  describe('Communication between <cds-radio-button-group> and <cds-radio-button>', () => {
    it('Should propagate disabled', async () => {
      render(
        template({ 'cds-radio-button-group': { disabled: true } }),
        document.body
      );
      await Promise.resolve();
      expect(
        Array.prototype.every.call(
          document.body.querySelectorAll('cds-radio-button'),
          (radio) => radio.disabled
        )
      ).toBe(true);
    });

    it('Should propagate labelPosition', async () => {
      render(
        template({
          'cds-radio-button-group': {
            labelPosition: RADIO_BUTTON_LABEL_POSITION.LEFT,
          },
        }),
        document.body
      );
      await Promise.resolve();
      expect(
        Array.prototype.every.call(
          document.body.querySelectorAll('cds-radio-button'),
          (radio) => radio.labelPosition === RADIO_BUTTON_LABEL_POSITION.LEFT
        )
      ).toBe(true);
    });

    it('Should propagate orientation', async () => {
      render(
        template({
          'cds-radio-button-group': {
            orientation: RADIO_BUTTON_ORIENTATION.VERTICAL,
          },
        }),
        document.body
      );
      await Promise.resolve();
      expect(
        Array.prototype.every.call(
          document.body.querySelectorAll('cds-radio-button'),
          (radio) => radio.orientation === RADIO_BUTTON_ORIENTATION.VERTICAL
        )
      ).toBe(true);
    });

    it('Should propagate name', async () => {
      render(
        template({
          'cds-radio-button-group': { name: 'name-foo' },
        }),
        document.body
      );
      await Promise.resolve();
      expect(
        Array.prototype.every.call(
          document.body.querySelectorAll('cds-radio-button'),
          (radio) => radio.name === 'name-foo'
        )
      ).toBe(true);
    });

    it('Should select <cds-radio-button> that matches the given value', async () => {
      render(
        template({
          'cds-radio-button-group': { value: 'staging' },
        }),
        document.body
      );
      await Promise.resolve();
      expect(
        Array.prototype.map.call(
          document.body.querySelectorAll('cds-radio-button'),
          (radio) => radio.checked
        )
      ).toEqual([false, false, true]);
    });

    it('Should update the value upon clicking <cds-radio-button>', async () => {
      render(
        template({
          'cds-radio-button-group': { name: 'name-foo' },
        }),
        document.body
      );
      await Promise.resolve();
      (
        document.body.querySelector(
          'cds-radio-button[value="staging"]'
        ) as HTMLElement
      ).click();
      expect(
        (
          document.body.querySelector(
            'cds-radio-button-group'
          ) as CDSRadioButtonGroup
        ).value
      ).toBe('staging');
    });

    it('Should update the value upon space key on <cds-radio-button>', async () => {
      render(
        template({
          'cds-radio-button-group': { name: 'name-foo' },
        }),
        document.body
      );
      await Promise.resolve();
      const event = new CustomEvent('keydown', {
        bubbles: true,
        composed: true,
      });
      const radioBaz = document.body.querySelector(
        'cds-radio-button[value="staging"]'
      );
      (radioBaz as HTMLElement).dispatchEvent(
        Object.assign(event, { key: ' ' })
      );
      expect(
        (
          document.body.querySelector(
            'cds-radio-button-group'
          ) as CDSRadioButtonGroup
        ).value
      ).toBe('staging');
    });

    it('Should update the value upon enter key on <cds-radio-button>', async () => {
      render(
        template({
          'cds-radio-button-group': { name: 'name-foo' },
        }),
        document.body
      );
      await Promise.resolve();
      const event = new CustomEvent('keydown', {
        bubbles: true,
        composed: true,
      });
      const radioBaz = document.body.querySelector(
        'cds-radio-button[value="staging"]'
      );
      (radioBaz as HTMLElement).dispatchEvent(
        Object.assign(event, { key: 'Enter' })
      );
      expect(
        (
          document.body.querySelector(
            'cds-radio-button-group'
          ) as CDSRadioButtonGroup
        ).value
      ).toBe('staging');
    });
  });

  describe('Keyboard navigation', () => {
    it('Should use left/right key for navigation in horizontal mode', async () => {
      render(
        template({
          'cds-radio-button-group': {
            orientation: RADIO_BUTTON_ORIENTATION.HORIZONTAL,
            name: 'name-foo',
          },
        }),
        document.body
      );
      await Promise.resolve();
      const radioFoo = document.body.querySelector(
        'cds-radio-button[value="all"]'
      ) as HTMLElement;
      radioFoo.focus();
      const event = new CustomEvent('keydown', {
        bubbles: true,
        composed: true,
      });
      radioFoo.dispatchEvent(Object.assign(event, { key: 'ArrowRight' }));
      expect(
        (
          document.body.querySelector(
            'cds-radio-button-group'
          ) as CDSRadioButtonGroup
        ).value
      ).toBe('cloudFoundry');
      expect(
        Array.prototype.map.call(
          document.body.querySelectorAll('cds-radio-button'),
          (radio) => radio.shadowRoot.querySelector('input').tabIndex
        )
      ).toEqual([-1, 0, -1]);
      const radioBar = document.body.querySelector(
        'cds-radio-button[value="cloudFoundry"]'
      ) as HTMLElement;
      radioBar.dispatchEvent(Object.assign(event, { key: 'ArrowLeft' }));
      expect(
        (
          document.body.querySelector(
            'cds-radio-button-group'
          ) as CDSRadioButtonGroup
        ).value
      ).toBe('all');
      expect(
        Array.prototype.map.call(
          document.body.querySelectorAll('cds-radio-button'),
          (radio) => radio.shadowRoot.querySelector('input').tabIndex
        )
      ).toEqual([0, -1, -1]);
    });

    it('Should use up/down key for navigation in vertical mode', async () => {
      render(
        template({
          'cds-radio-button-group': {
            orientation: RADIO_BUTTON_ORIENTATION.VERTICAL,
            name: 'name-foo',
          },
        }),
        document.body
      );
      await Promise.resolve();
      const radioFoo = document.body.querySelector(
        'cds-radio-button[value="all"]'
      ) as HTMLElement;
      radioFoo.focus();
      const event = new CustomEvent('keydown', {
        bubbles: true,
        composed: true,
      });
      radioFoo.dispatchEvent(Object.assign(event, { key: 'ArrowDown' }));
      expect(
        (
          document.body.querySelector(
            'cds-radio-button-group'
          ) as CDSRadioButtonGroup
        ).value
      ).toBe('cloudFoundry');
      expect(
        Array.prototype.map.call(
          document.body.querySelectorAll('cds-radio-button'),
          (radio) => radio.shadowRoot.querySelector('input').tabIndex
        )
      ).toEqual([-1, 0, -1]);
      const radioBar = document.body.querySelector(
        'cds-radio-button[value="cloudFoundry"]'
      ) as HTMLElement;
      radioBar.dispatchEvent(Object.assign(event, { key: 'ArrowUp' }));
      expect(
        (
          document.body.querySelector(
            'cds-radio-button-group'
          ) as CDSRadioButtonGroup
        ).value
      ).toBe('all');
      expect(
        Array.prototype.map.call(
          document.body.querySelectorAll('cds-radio-button'),
          (radio) => radio.shadowRoot.querySelector('input').tabIndex
        )
      ).toEqual([0, -1, -1]);
    });
  });

  describe('Event-based form participation', () => {
    it('Should respond to `formdata` event', async () => {
      render(
        html`
          <form>
            ${template({
              'cds-radio-button-group': {
                name: 'name-foo',
                value: 'staging',
              },
            })}
          </form>
        `,
        document.body
      );
      await Promise.resolve();
      const formData = new FormData();
      const event = new CustomEvent('formdata', {
        bubbles: true,
        cancelable: false,
        composed: false,
      }) as unknown as FormDataEventLike;
      event.formData = formData;
      const form = document.querySelector('form');
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion -- https://github.com/carbon-design-system/carbon/issues/20452
      form!.dispatchEvent(event);
      expect(getValues(formData)).toEqual({ 'name-foo': 'staging' });
    });

    it('Should not respond to `formdata` event if form item name is not specified', async () => {
      render(
        html`
          <form>
            ${template({
              'cds-radio-button-group': {
                value: 'staging',
              },
            })}
          </form>
        `,
        document.body
      );
      await Promise.resolve();
      const formData = new FormData();
      const event = new CustomEvent('formdata', {
        bubbles: true,
        cancelable: false,
        composed: false,
      }) as unknown as FormDataEventLike;
      event.formData = formData;
      const form = document.querySelector('form');
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion -- https://github.com/carbon-design-system/carbon/issues/20452
      form!.dispatchEvent(event);
      expect(getValues(formData)).toEqual({});
    });

    it('Should not respond to `formdata` event if no item is selected', async () => {
      render(
        html`
          <form>
            ${template({
              'cds-radio-button-group': {
                name: 'name-foo',
              },
            })}
          </form>
        `,
        document.body
      );
      await Promise.resolve();
      const formData = new FormData();
      const event = new CustomEvent('formdata', {
        bubbles: true,
        cancelable: false,
        composed: false,
      }) as unknown as FormDataEventLike;
      event.formData = formData;
      const form = document.querySelector('form');
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion -- https://github.com/carbon-design-system/carbon/issues/20452
      form!.dispatchEvent(event);
      expect(getValues(formData)).toEqual({});
    });

    it('Should not respond to `formdata` event if disabled', async () => {
      render(
        html`
          <form>
            ${template({
              'cds-radio-button-group': {
                disabled: true,
                name: 'name-foo',
              },
            })}
          </form>
        `,
        document.body
      );
      await Promise.resolve();
      const formData = new FormData();
      const event = new CustomEvent('formdata', {
        bubbles: true,
        cancelable: false,
        composed: false,
      }) as unknown as FormDataEventLike;
      event.formData = formData;
      expect(getValues(formData)).toEqual({});
    });
  });

  afterEach(async () => {
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion -- https://github.com/carbon-design-system/carbon/issues/20452
    render(undefined!, document.body);
    await Promise.resolve();
  });
});
